<?php

namespace App\Http\Controllers;

use App\Config;
use App\Menu;
use App\Package;
use App\Movie;
use App\TvSeries;
use App\PpvPurchase;
use App\Gst;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\URL;
use PayPal\Api\Amount;
use PayPal\Api\Details;
use PayPal\Api\Item;
use PayPal\Api\ItemList;
use PayPal\Api\Payer;
use PayPal\Api\Payment;
use PayPal\Api\PaymentExecution;
use PayPal\Api\RedirectUrls;
use PayPal\Api\Transaction;
use PayPal\Auth\OAuthTokenCredential;
use PayPal\Rest\ApiContext;
use Redirect;


class PaypalController extends Controller
{
    
    private $_api_context;
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        /** setup PayPal api context **/
        $paypal_conf = \Config::get('paypal');
        $this->_api_context = new ApiContext(new OAuthTokenCredential($paypal_conf['client_id'], $paypal_conf['secret']));
        $this->_api_context->setConfig($paypal_conf['settings']);
    }
    
    /**
     * Store a details of payment with paypal.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function postPaymentWithpaypal(Request $request)
    {
        $currency_code = Config::first()->currency_code;
        $currency_code = strtoupper($currency_code);
        $paymentType = $request->payment_type ?? 'subscription';
        $gst = Gst::first();
        
        $payer = new Payer();
        $payer->setPaymentMethod('paypal');
        
        // Initialize variables for use in different payment types
        $item_1 = new Item();
        $amount_coupon = 0;
        $item_name = '';
        $item_price = 0;
        
        // Handle different payment types
        if ($paymentType == 'subscription') {
            // Handle subscription payment
            $plan = Package::findOrFail($request->plan_id);
            if (Session::has('coupon_applied')) {
                $amount_coupon = $plan['amount'] - Session::get('coupon_applied')['amount'];
            } else {
                $amount_coupon = $plan['amount'];
            }
            
            $item_name = $plan->name;
            $item_price = $amount_coupon;
            
            // Store plan in session for later use
            Session::put('plan', $plan);
            Session::put('payment_type', 'subscription');
            
        } elseif ($paymentType == 'ppv') {
            // Handle PPV payment for movies
            $movie = Movie::findOrFail($request->item_id);
            
            // Calculate price with GST if applicable
            if ($movie->offer_price != null) {
                $item_price = $movie->offer_price;
                if ($gst && $gst->status == 1) {
                    $gst_amount = ($movie->offer_price * $gst->gst_per) / 100;
                    $item_price = $movie->offer_price + $gst_amount;
                }
            } else {
                $item_price = $movie->price;
                if ($gst && $gst->status == 1) {
                    $gst_amount = ($movie->price * $gst->gst_per) / 100;
                    $item_price = $movie->price + $gst_amount;
                }
            }
            
            $item_name = "Movie: " . $movie->title;
            
            // Store movie info in session
            Session::put('ppv_item', $movie);
            Session::put('payment_type', 'ppv');
            Session::put('ppv_type', 'movie');
            
        } elseif ($paymentType == 'rental_series') {
            // Handle rental payment for TV series
            $series = Season::findOrFail($request->item_id);
            
            // Calculate price with GST if applicable
            if ($series->offer_price != null) {
                $item_price = $series->offer_price;
                if ($gst && $gst->status == 1) {
                    $gst_amount = ($series->offer_price * $gst->gst_per) / 100;
                    $item_price = $series->offer_price + $gst_amount;
                }
            } else {
                $item_price = $series->price;
                if ($gst && $gst->status == 1) {
                    $gst_amount = ($series->price * $gst->gst_per) / 100;
                    $item_price = $series->price + $gst_amount;
                }
            }
            
            $item_name = "Series: " . $series->title;
            
            // Store series info in session
            Session::put('ppv_item', $series);
            Session::put('payment_type', 'rental_series');
            Session::put('ppv_type', 'series');
        }
        
        // Set up item details
        $item_1->setName($item_name)
            ->setCurrency('USD')
            ->setQuantity(1)
            ->setPrice($item_price);
            
        $item_list = new ItemList();
        $item_list->setItems(array($item_1));
        
        $amount = new Amount();
        $amount->setCurrency('USD')
            ->setTotal($item_price);
        
        $transaction = new Transaction();
        $transaction->setAmount($amount)
            ->setItemList($item_list)
            ->setDescription($paymentType == 'subscription' ? 'Subscription' : 'Pay Per View');
        
        $redirect_urls = new RedirectUrls();
        $redirect_urls->setReturnUrl(route('getPaymentStatus'))
            ->setCancelUrl(route('getPaymentFailed'));
        
        $payment = new Payment();
        $payment->setIntent('Sale')
            ->setPayer($payer)
            ->setRedirectUrls($redirect_urls)
            ->setTransactions(array($transaction));
        
        try {
           $payment->create($this->_api_context);
        } catch (\PayPal\Exception\PPConnectionException $ex) {
            if (\Config::get('app.debug')) {
                return back()->with('deleted', __('Connection timeout'));
            } else {
                return back()->with('deleted', __('Some error occur, sorry for inconvenient'));
            }
        }
        
        foreach ($payment->getLinks() as $link) {
            if ($link->getRel() == 'approval_url') {
                $redirect_url = $link->getHref();
                break;
            }
        }
        
        // Add payment ID to session
        Session::put('paypal_payment_id', $payment->getId());
        
        if (isset($redirect_url)) {
            // Redirect to PayPal
            return redirect($redirect_url);
        }
        
        return back()->with('deleted', __('Unknown error occurred'));
    }

    public function getPaymentStatus(Request $request)
    {
        $user_email = Auth::user()->email;
        $com_email = Config::findOrFail(1)->w_email;

        Session::put('user_email', $user_email);
        Session::put('com_email', $com_email);

        // Get the payment ID before session clear
        $payment_id = Session::get('paypal_payment_id');
        $payment_type = Session::get('payment_type');
        
        // Handle different payment types
        if ($payment_type == 'subscription') {
            $plan = Session::get('plan');
            if (Session::has('coupon_applied')) {
                $amount_coupon = $plan['amount'] - Session::get('coupon_applied')['amount'];
            } else {
                $amount_coupon = $plan['amount'];
            }
            $payment_amount = $amount_coupon;
        } else {
            // For PPV or rental series
            $ppv_item = Session::get('ppv_item');
            $payment_amount = $ppv_item->offer_price ?? $ppv_item->price;
        }
        
        // Clear the session payment ID
        Session::forget('paypal_payment_id');
        
        if (empty($request->get('PayerID')) || empty($request->get('token'))) {
            return back()->with('deleted', __('Payment failed'));
        }

        $payment = Payment::get($payment_id, $this->_api_context);
        $execution = new PaymentExecution();
        $execution->setPayerId($request->get('PayerID'));
        
        // Execute the payment
        $result = $payment->execute($execution, $this->_api_context);
        
        if ($result->getState() == 'approved') {
            // Payment successful - process based on type
            $payment_method = 'paypal';
            $payment_status = 1;
            
            if ($payment_type == 'subscription') {
                // Handle subscription payment
                $plan_id = $plan->id;
                Session::forget('plan');
                
                $checkout = new SubscriptionController;
                return $checkout->subscribe($payment_id, $payment_method, $plan_id, $payment_status, $payment_amount);
                
            } elseif ($payment_type == 'ppv') {
                // Handle PPV payment for movies
                $movie = $ppv_item;
                $gst = Gst::first();
                $hours = $movie->hours;
                $expiresAt = now()->addHours($hours);
                
                // Calculate price with GST
                $gst_percentage = 0;
                if ($gst && $gst->status == 1) {
                    $gst_percentage = $gst->gst_per;
                }
                
                // Save PPV purchase in the database
                $purchase = PpvPurchase::create([
                    'user_id' => auth()->id(),
                    'movie_id' => $movie->id,
                    'payment_id' => $payment_id,
                    'gst' => $gst_percentage,
                    'price' => $movie->price,
                    'offer_price' => $movie->offer_price ?? $movie->price,
                    'hours' => $hours,
                    'expires_at' => $expiresAt,
                    'type' => 'movie',
                    'payment_type' => 'PayPal',
                    'status' => 1, // Active status
                ]);
                
                Session::forget('ppv_item');
                Session::forget('payment_type');
                Session::forget('ppv_type');
                
                if (getSubscription()->getData()->subscribed == true) {
                    // Subscription is active
                    $route = 'movie/detail/';
                } else {
                    // Subscription is not active
                    $route = 'movie/guest/detail/';
                }
                
                return redirect()->to($route . $movie->slug)->with('success', __('Movie purchase successful!'));
                
            } elseif ($payment_type == 'rental_series') {
                // Handle rental payment for TV series
                $series = $ppv_item;
                $gst = Gst::first();
                $hours = $series->hours;
                $expiresAt = now()->addHours($hours);
                
                // Calculate price with GST
                $gst_percentage = 0;
                if ($gst && $gst->status == 1) {
                    $gst_percentage = $gst->gst_per;
                }
                
                // Save PPV purchase in the database
                $purchase = PpvPurchase::create([
                    'user_id' => auth()->id(),
                    'tv_series_id' => $series->id,
                    'payment_id' => $payment_id,
                    'gst' => $gst_percentage,
                    'price' => $series->price,
                    'offer_price' => $series->offer_price ?? $series->price,
                    'hours' => $hours,
                    'expires_at' => $expiresAt,
                    'type' => 'series',
                    'payment_type' => 'PayPal',
                    'status' => 1, // Active status
                ]);
                
                Session::forget('ppv_item');
                Session::forget('payment_type');
                Session::forget('ppv_type');
                
                if (getSubscription()->getData()->subscribed == true) {
                    // Subscription is active
                    $route = 'show/detail/';
                } else {
                    // Subscription is not active
                    $route = 'show/guest/detail/';
                }
                
                return redirect()->to($route . $series->seasons_first->season_slug)->with('success', __('Series rental successful!'));
            }
            
        } else {
            return redirect('/')->with('deleted', __('Payment failed'));
        }
    }

    public function getPaymentFailed()
    {
        return redirect('/')->with('deleted', __('Payment failed'));
    }

}